<?php
require_once 'Mail.php';
require_once 'Mail/mime.php';

/**
 *　かんたんメール送信クラス
 *
 *　　Ver 1.02  2011/9/14  サンプルコードにミスがあったのを修正しました。
 *　　(個人・商用ともにフリー利用可。ご自由にお使いください。無償提供のため、お問い合わせはご遠慮ください。)
 *
 *　　亜洲トークン株式会社
 *　　http://www.ashutoken.jp/
 *
 *　　ちかともモバイル
 *　　http://m.chikatomo.jp/
 *
 */
class SmtpMailSender
{
	private $smtp				= null;

	private $_fromName			= "";
	private $_fromMailAddress	= "";
	private $_toName			= "";
	private $_toMailAddress		= "";
	private $_subject			= "";
	private $_body				= "";

	private $crlf				= "\n";	// 現在の改行コード

	/**
	 *	コンストラクタ
	 *
	 *	@param	string		$host		接続先サーバ
	 *	@param	integer		$port		接続先ポート
	 *	@param	string		$user		認証ユーザー名
	 *	@param	string		$pass		認証パス
	 *	@param	string 		$localhost	EHLO あるいは HELO を送信する際に用いられる値 
	 *	@param	integer		$timeOut	接続タイムアウト時間（秒）
	 */
	public function __construct($host, $port, $user = "", $pass = "", $localhost = "localhost", $timeOut = null)
	{
		$params = array();
		$params["host"]			= $host;
		$params["port"]			= $port;
		if (strlen($user) > 0) {
			$params["auth"]			= TRUE;
			$params["username"]		= $user;
			$params["password"]		= $pass;
		} else {
			$params["auth"]			= FALSE;
			$params["username"]		= null;
			$params["password"]		= null;
		}
		$params["localhost"]	= $localhost;
		$params["timeout"]		= $timeOut;
		$params["debug"]		= FALSE;
		$params["persist"]		= null;

		$this->smtp =& Mail::factory('smtp', $params);
	}


	/**
	 *	入力のチェック
	 */
    public function checkInputData()
    {
		// 入力項目が１つでも空なら
		if (	(strlen($this->_fromName) == 0)			||
				(strlen($this->_fromMailAddress) == 0)	||
				(strlen($this->_toName) == 0)			||
				(strlen($this->_toMailAddress) == 0)	||
				(strlen($this->_subject) == 0)			||
				(strlen($this->_body) == 0)					) {
			return false;
		}

		// メールアドレスの形式が正しくなければ
		$regexp = '/[!#$%&\'*+-.\/0-9=?A-Z^_`a-z{|}~]+@(?:[0-9A-Za-z][-0-9A-Za-z.]*[0-9A-Za-z]|\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])/';
		if (	(preg_match($regexp, $this->_fromMailAddress) == 0)	||
				(preg_match($regexp, $this->_toMailAddress) == 0)		) {
			return false;
		}
		return true;
    }


	/**
	 *	メールを送信する
	 *
	 *	@return	boolean
	 */
	public function send()
	{
		// 入力チェック
		if (!$this->checkInputData()) {
			throw new Exception("送信エラー:入力チェック時");
		}

		$recipients			= $this->_toMailAddress;
		$headers['From']	= mb_encode_mimeheader($this->_fromName) . " <" . $this->_fromMailAddress . ">";
		$headers['To']		= mb_encode_mimeheader($this->_toName) . " <" . $this->_toMailAddress . ">";
		$headers['Subject']	= $this->_subject;

		// mimeインスタンス生成
		$mime =& new Mail_mime($this->crlf);
		$mime->setTXTBody($this->_body);

		// 出力用パラメータ
		$build_params = array(
			"html_charset" => "EUC-JP",
			"text_charset" => "ISO-2022-JP",
			"head_charset" => "ISO-2022-JP",
		);
		$body		= $mime->get($build_params);
		$headers	= $mime->headers($headers);

		$result = $this->smtp->send($recipients, $headers, $body);
		if (PEAR::isError($result)) {
			throw new Exception($result->getMessage());
		}

		// ok
		return true;
	}


	/**
	 *	送信データ用メンバ変数の全てクリアする
	 */
    public function clear()
    {
		$this->_fromName			= "";
		$this->_fromMailAddress		= "";
		$this->_toName				= "";
		$this->_toMailAddress		= "";
		$this->_subject				= "";
		$this->_body				= "";
    }


	/// セッター群
    /// 送信者の名前とメールアドレスを設定する
	public function setFrom($name, $mailAddress)
	{
		$this->_fromName			= $name;
		$this->_fromMailAddress		= $mailAddress;
	}

    /// 送り先の名前とメールアドレスを設定する
	public function setTo($name, $mailAddress)
	{
		$this->_toName				= $name;
		$this->_toMailAddress		= $mailAddress;
	}

    /// メールの表題を設定する
	public function setSubject($value)
	{
		// UTF-8 or EUC-JP => JIS
		$original	= mb_internal_encoding();
		$value		= mb_convert_encoding( $value, "ISO-2022-JP", "UTF-8, EUC-JP" );
		mb_internal_encoding( "ISO-2022-JP" );
		$value		= mb_encode_mimeheader( $value, "ISO-2022-JP" );
		mb_internal_encoding( $original );

		$this->_subject	= $value;
	}

    /// メールの本文を設定する
	public function setBody($value)
	{
		$this->_body	= mb_convert_encoding( $value, "ISO-2022-JP", "UTF-8, EUC-JP" );
	}

	/**
	 *	テンプレートから送信データの本文を設定する
	 *
	 *	@param	string		$templateFile	テンプレートファイル名
	 *	@param	array		$messageParams	テンプレートパラメータの配列
	 */
	public function setBodyFromTemplate($templateFile, $messageParams)
	{
		// テンプレートファイルの読み込み
		if (!file_exists($templateFile)) {
			throw new Exception("送信エラー:テンプレートファイルが存在しません");
		}
		$contents = file_get_contents($templateFile);

		// テンプレート内容と値の配列から生成した本文をセット
		$this->setBody(vsprintf($contents, $messageParams));
	}

}
?>